from odoo import models, fields, api
from datetime import datetime
from io import BytesIO
import base64
import csv

class StockCardReport(models.TransientModel):
    _name = 'stock.card.report'
    _description = 'Stock Card Report'

    # Report criteria fields
    date_from = fields.Date(string='Start Date', required=True)
    date_to = fields.Date(string='End Date', required=True)
    product_ids = fields.Many2many('product.product', string='Products')
    category_ids = fields.Many2many('product.category', string='Categories')
    location_id = fields.Many2one('stock.location', string='Location')
    company_id = fields.Many2one('res.company', string='Company')

    # Add new field to store Excel file
    excel_file = fields.Binary('Excel Report', readonly=True)
    excel_filename = fields.Char('Excel File Name', readonly=True)

    def _get_stock_card_lines(self):
        self.ensure_one()
        domain = [
            ('date', '>=', self.date_from),
            ('date', '<=', self.date_to),
            ('location_id', '=', self.location_id.id),
            ('company_id', '=', self.company_id.id),
        ]
        
        if self.product_ids:
            domain.append(('product_id', 'in', self.product_ids.ids))
        elif self.category_ids:
            domain.append(('product_id.categ_id', 'in', self.category_ids.ids))

        moves = self.env['stock.move.line'].search(domain, order='date')
        
        lines = []
        balance = 0
        for move in moves:
            quantity_in = move.qty_done if move.location_dest_id == self.location_id else 0
            quantity_out = move.qty_done if move.location_id == self.location_id else 0
            balance += quantity_in - quantity_out
            
            lines.append({
                'date': move.date,
                'reference': move.reference or move.picking_id.name,
                'quantity_in': quantity_in,
                'quantity_out': quantity_out,
                'balance': balance,
            })
        
        return lines 

    def generate_excel_report(self, product_id, location_id, start_date, end_date):
        # Create Excel file in memory
        output = BytesIO()
        
        # Write headers
        headers = ['Date', 'Reference', 'Partner', 'Product', 'In Qty', 'Out Qty', 'Balance']
        worksheet_data = [headers]
        
        # Get stock card lines
        lines = self._get_stock_card_lines(product_id, location_id, start_date, end_date)
        
        # Add data rows
        for line in lines:
            row = [
                line['date'].strftime('%Y-%m-%d %H:%M:%S'),
                line.get('reference', ''),
                line.get('partner_name', ''),
                line.get('product_name', ''),
                line.get('in_qty', 0.0),
                line.get('out_qty', 0.0),
                line.get('balance', 0.0),
            ]
            worksheet_data.append(row)
        
        # Write to Excel format
        def write_csv_to_binary():
            csv_data = BytesIO()
            for row in worksheet_data:
                csv_data.write(','.join(str(cell) for cell in row).encode())
                csv_data.write(b'\n')
            return csv_data.getvalue()
        
        excel_binary = write_csv_to_binary()
        
        # Convert to base64
        excel_base64 = base64.b64encode(excel_binary)
        
        return excel_base64

    @api.model
    def get_report_values(self, docids, data=None):
        # ... existing code ...
        
        # Generate Excel file
        excel_content = self.generate_excel_report(
            data['product_id'],
            data['location_id'],
            data['start_date'],
            data['end_date']
        )
        
        # Update the record with Excel file
        self.write({
            'excel_file': excel_content,
            'excel_filename': f'stock_card_report_{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}.csv'
        })
        
        # Return existing report values
        return {
            'doc_ids': data.get('ids', []),
            'doc_model': 'product.product',
            'docs': self.env['product.product'].browse(data['product_id']),
            'data': data,
            'lines': self._get_stock_card_lines(
                data['product_id'],
                data['location_id'],
                data['start_date'],
                data['end_date']
            ),
        }

    def action_export_excel(self):
        self.ensure_one()
        lines = self._get_stock_card_lines()
        
        # Create CSV in memory
        output = BytesIO()
        writer = csv.writer(output)
        
        # Write headers
        headers = ['Date', 'Reference', 'In Qty', 'Out Qty', 'Balance']
        writer.writerow(headers)
        
        # Write data rows
        for line in lines:
            row = [
                line['date'].strftime('%Y-%m-%d %H:%M:%S'),
                line['reference'],
                line['quantity_in'],
                line['quantity_out'],
                line['balance'],
            ]
            writer.writerow(row)
        
        # Encode the CSV data
        excel_data = base64.b64encode(output.getvalue().encode())
        
        # Generate filename
        filename = f'stock_card_report_{datetime.now().strftime("%Y%m%d_%H%M%S")}.csv'
        
        # Save the file
        self.write({
            'excel_file': excel_data,
            'excel_filename': filename,
        })
        
        # Return download action
        return {
            'type': 'ir.actions.act_url',
            'url': f'/web/content/?model={self._name}&id={self.id}&field=excel_file&filename={filename}&download=true',
            'target': 'self',
        }