from odoo import models, fields, api



from odoo.exceptions import ValidationError



from datetime import datetime, date



from odoo.tools.translate import _







class StockCardWizard(models.TransientModel):



    _name = 'stock.card.wizard'



    _description = 'Stock Card Report Wizard'







    start_date = fields.Date(string='Start Date', required=True, 



                           default=lambda self: date(date.today().year, date.today().month, 1))



    end_date = fields.Date(string='End Date', required=True,



                         default=fields.Date.today)



    company_id = fields.Many2one('res.company', string='Company', required=True,



                               default=lambda self: self.env.company)



    warehouse_id = fields.Many2one('stock.warehouse', string='Warehouse', required=True,



                                 default=lambda self: self.env['stock.warehouse'].search([('company_id', '=', self.env.company.id)], limit=1))



    location_id = fields.Many2one(



        'stock.location', 



        string='Location', 



        required=True,



        domain="[('usage', '=', 'internal'), ('warehouse_id', '=', warehouse_id)]",



        default=lambda self: self._get_default_location()



    )



    report_by = fields.Selection([



        ('product', 'Product'),



        ('category', 'Product Category')



    ], string='Report By', required=True, default='product')



    



    product_ids = fields.Many2many('product.product', string='Products',
                                 domain="[('type', '=', 'consu')]")



    category_ids = fields.Many2many('product.category', string='Product Categories')



    printed_by = fields.Many2one('res.users', string='Printed By', 



                                default=lambda self: self.env.user, readonly=True)







    @api.model



    def _get_default_location(self):



        warehouse = self.env['stock.warehouse'].search([('company_id', '=', self.env.company.id)], limit=1)



        if warehouse:



            return warehouse.lot_stock_id.id



        return False







    @api.onchange('warehouse_id')



    def _onchange_warehouse_id(self):



        if self.warehouse_id:



            domain = [



                ('usage', '=', 'internal'),



                ('warehouse_id', '=', self.warehouse_id.id)



            ]



            location = self.env['stock.location'].search(domain, limit=1)



            self.location_id = location or self.warehouse_id.lot_stock_id



        else:



            self.location_id = False







    @api.onchange('report_by')



    def _onchange_report_by(self):



        self.product_ids = False



        self.category_ids = False







    @api.constrains('start_date', 'end_date')



    def _check_dates(self):



        for record in self:



            if record.start_date > record.end_date:



                raise ValidationError(_('Start Date must be before End Date'))







    def action_print_pdf(self):



        self.ensure_one()



        if self.report_by == 'product' and not self.product_ids:



            raise ValidationError(_('Please select at least one product.'))



        if self.report_by == 'category' and not self.category_ids:



            raise ValidationError(_('Please select at least one category.'))



        return self.env.ref('sss_stock_card_report.action_stock_card_report').report_action(self) 




    def action_export_excel(self):
        self.ensure_one()
        
        if self.report_by == 'product' and not self.product_ids:
            raise ValidationError(_('Please select at least one product.'))
        
        if self.report_by == 'category' and not self.category_ids:
            raise ValidationError(_('Please select at least one category.'))
        
        # Try to use Excel report if available
        try:
            excel_report = self.env.ref('sss_stock_card_report.stock_card_xlsx_report', raise_if_not_found=False)
            if excel_report:
                return excel_report.report_action(self)
        except:
            pass
        
        # Fallback: Generate Excel using xlsxwriter
        try:
            import xlsxwriter
            from io import BytesIO
            import base64
            
            output = BytesIO()
            workbook = xlsxwriter.Workbook(output, {'in_memory': True})
            
            # Get report data
            report_model = self.env['report.sss_stock_card_report.stock_card_report']
            report_data = report_model._get_report_data(self)
            
            # Format definitions
            header_format = workbook.add_format({
                'bold': True,
                'align': 'center',
                'valign': 'vcenter',
                'fg_color': '#875A7B',
                'color': 'white',
                'border': 1,
            })
            
            date_format = workbook.add_format({
                'num_format': 'yyyy-mm-dd',
                'align': 'center',
            })
            
            number_format = workbook.add_format({
                'num_format': '#,##0.00',
                'align': 'right',
            })
            
            positive_format = workbook.add_format({
                'num_format': '#,##0.00',
                'align': 'right',
                'color': 'green',
            })
            
            negative_format = workbook.add_format({
                'num_format': '#,##0.00',
                'align': 'right',
                'color': 'red',
            })
            
            total_format = workbook.add_format({
                'bold': True,
                'num_format': '#,##0.00',
                'align': 'right',
                'fg_color': '#875A7B',
                'color': 'white',
                'border': 1,
            })
            
            # Create worksheet for each product
            for product_data in report_data:
                sheet_name = product_data['product'].display_name[:31]  # Excel sheet name limit
                worksheet = workbook.add_worksheet(sheet_name)
                
                # Set column widths
                worksheet.set_column(0, 0, 15)  # Date
                worksheet.set_column(1, 1, 20)  # Origin
                worksheet.set_column(2, 4, 15)  # Quantities and Balance
                
                # Write headers
                headers = ['Date', 'Origin', 'Quantity In', 'Quantity Out', 'Balance']
                for col, header in enumerate(headers):
                    worksheet.write(0, col, header, header_format)
                
                # Write data
                row = 1
                for move in product_data['moves']:
                    worksheet.write(row, 0, move['date'], date_format)
                    worksheet.write(row, 1, move.get('origin', ''))
                    worksheet.write(row, 2, move.get('quantity_in', 0), 
                                  positive_format if move.get('quantity_in', 0) > 0 else number_format)
                    worksheet.write(row, 3, move.get('quantity_out', 0),
                                  negative_format if move.get('quantity_out', 0) > 0 else number_format)
                    balance = move.get('balance', 0)
                    worksheet.write(row, 4, balance,
                                  positive_format if balance >= 0 else negative_format)
                    row += 1
                
                # Write totals
                worksheet.write(row, 1, 'Total', header_format)
                total_in = sum(m.get('quantity_in', 0) for m in product_data['moves'])
                total_out = sum(m.get('quantity_out', 0) for m in product_data['moves'])
                final_balance = product_data['moves'][-1].get('balance', 0) if product_data['moves'] else 0
                worksheet.write(row, 2, total_in, total_format)
                worksheet.write(row, 3, total_out, total_format)
                worksheet.write(row, 4, final_balance, total_format)
            
            workbook.close()
            output.seek(0)
            
            # Create attachment
            filename = f'stock_card_report_{datetime.now().strftime("%Y%m%d_%H%M%S")}.xlsx'
            attachment = self.env['ir.attachment'].create({
                'name': filename,
                'datas': base64.b64encode(output.getvalue()).decode('utf-8'),
                'res_model': self._name,
                'res_id': self.id,
            })
            
            # Return download action
            return {
                'type': 'ir.actions.act_url',
                'url': f'/web/content/{attachment.id}?download=true',
                'target': 'self',
            }
        except ImportError:
            raise ValidationError(_('xlsxwriter library is required for Excel export. Please install it.'))
        except Exception as e:
            raise ValidationError(_('Error generating Excel report: %s') % str(e))







    def _get_low_stock_threshold(self, product):



        return product.stock_card_low_threshold



    def _get_normal_stock_threshold(self, product):



        return product.stock_card_normal_threshold




















































































































































