# -*- coding: utf-8 -*-
from odoo import models
from datetime import datetime

class StockCardXlsx(models.AbstractModel):
    _name = 'report.sss_stock_card_report.stock_card_xlsx'
    _inherit = 'report.report_xlsx.abstract'
    _description = 'Stock Card Excel Report'

    def _get_report_name(self, wizard):
        return f'Stock Card Report - {datetime.now().strftime("%Y-%m-%d %H-%M-%S")}'

    def _get_report_columns(self, report):
        return {
            0: {'header': 'Date', 'field': 'date', 'width': 15},
            1: {'header': 'Origin', 'field': 'origin', 'width': 20},
            2: {'header': 'Quantity In', 'field': 'quantity_in', 'width': 15, 'type': 'number'},
            3: {'header': 'Quantity Out', 'field': 'quantity_out', 'width': 15, 'type': 'number'},
            4: {'header': 'Balance', 'field': 'balance', 'width': 15, 'type': 'number'},
        }

    def generate_xlsx_report(self, workbook, data, wizard):
        report_model = self.env['report.sss_stock_card_report.stock_card_report']
        report_data = report_model._get_report_data(wizard)

        # Format definitions
        header_format = workbook.add_format({
            'bold': True,
            'align': 'center',
            'valign': 'vcenter',
            'fg_color': '#875A7B',
            'color': 'white',
            'border': 1,
        })

        date_format = workbook.add_format({
            'num_format': 'yyyy-mm-dd',
            'align': 'center',
        })

        number_format = workbook.add_format({
            'num_format': '#,##0.00',
            'align': 'right',
        })

        positive_number = workbook.add_format({
            'num_format': '#,##0.00',
            'align': 'right',
            'color': 'green',
        })

        negative_number = workbook.add_format({
            'num_format': '#,##0.00',
            'align': 'right',
            'color': 'red',
        })

        total_format = workbook.add_format({
            'bold': True,
            'num_format': '#,##0.00',
            'align': 'right',
            'fg_color': '#875A7B',
            'color': 'white',
            'border': 1,
        })

        # Get columns info
        columns = self._get_report_columns(report_data)

        for product_data in report_data:
            # Create worksheet for each product
            sheet_name = product_data['product'].display_name[:31]
            sheet = workbook.add_worksheet(sheet_name)

            # Set columns width
            for col, column in columns.items():
                sheet.set_column(col, col, column['width'])

            # Write headers
            for col, column in columns.items():
                sheet.write(0, col, column['header'], header_format)

            # Write data
            row = 1
            for move in product_data['moves']:
                sheet.write(row, 0, move['date'], date_format)
                sheet.write(row, 1, move['origin'])
                sheet.write(row, 2, move['quantity_in'], positive_number if move['quantity_in'] else number_format)
                sheet.write(row, 3, move['quantity_out'], negative_number if move['quantity_out'] else number_format)
                sheet.write(row, 4, move['balance'], 
                          positive_number if move['balance'] >= 0 else negative_number)
                row += 1

            # Write totals
            sheet.write(row, 1, 'Total', header_format)
            sheet.write(row, 2, sum(m['quantity_in'] for m in product_data['moves']), total_format)
            sheet.write(row, 3, sum(m['quantity_out'] for m in product_data['moves']), total_format)
            sheet.write(row, 4, product_data['moves'][-1]['balance'], total_format)

            # Write stock status
            row += 2
            sheet.write(row, 0, 'Stock Status:', header_format)
            sheet.write(row, 1, product_data['stock_status']['text'])
            sheet.write(row, 2, f"Low Threshold: {product_data['product'].stock_card_low_threshold}")
            sheet.write(row + 1, 2, f"Normal Threshold: {product_data['product'].stock_card_normal_threshold}")

            # Write stock status
            row += 2
            sheet.write(row, 0, 'Stock Status:', header_format)
            sheet.write(row, 1, product_data['stock_status']['text'])
            sheet.write(row, 2, f"Low Threshold: {wizard._get_low_stock_threshold()}") 