from odoo import models, fields, api
from odoo.tools.misc import get_lang
from datetime import datetime
from io import BytesIO
import base64
import csv

class StockCardWizard(models.TransientModel):
    _name = 'stock.card.wizard'
    _description = 'Stock Card Wizard'

    start_date = fields.Date(string='Start Date', required=True)
    end_date = fields.Date(string='End Date', required=True)
    company_id = fields.Many2one('res.company', string='Company', required=True, 
                                default=lambda self: self.env.company)
    warehouse_id = fields.Many2one('stock.warehouse', string='Warehouse', required=True)
    location_id = fields.Many2one('stock.location', string='Location')
    report_by = fields.Selection([
        ('product', 'Product'),
        ('category', 'Category')
    ], string='Report By', default='product', required=True)
    product_ids = fields.Many2many('product.product', string='Products')
    category_ids = fields.Many2many('product.category', string='Categories')
    printed_by = fields.Many2one('res.users', string='Printed By', 
                                default=lambda self: self.env.user, readonly=True)

    @api.onchange('warehouse_id')
    def _onchange_warehouse_id(self):
        if self.warehouse_id:
            self.location_id = self.warehouse_id.lot_stock_id.id

    def action_export_excel(self):
        self.ensure_one()
        
        # Get report data
        report = self.env['stock.card.report'].create({
            'date_from': self.start_date,
            'date_to': self.end_date,
            'location_id': self.location_id.id,
            'company_id': self.company_id.id,
            'product_ids': [(6, 0, self.product_ids.ids)] if self.report_by == 'product' else False,
            'category_ids': [(6, 0, self.category_ids.ids)] if self.report_by == 'category' else False,
        })
        
        lines = report._get_stock_card_lines()
        
        # Create CSV in memory
        output = BytesIO()
        writer = csv.writer(output)
        
        # Write headers
        headers = ['Date', 'Reference', 'In Qty', 'Out Qty', 'Balance']
        writer.writerow(headers)
        
        # Write data rows
        for line in lines:
            row = [
                line['date'].strftime('%Y-%m-%d %H:%M:%S'),
                line['reference'],
                line['quantity_in'],
                line['quantity_out'],
                line['balance'],
            ]
            writer.writerow(row)
        
        # Encode the CSV data
        excel_data = base64.b64encode(output.getvalue().encode())
        
        # Generate filename
        filename = f'stock_card_report_{datetime.now().strftime("%Y%m%d_%H%M%S")}.csv'
        
        # Create attachment
        attachment = self.env['ir.attachment'].create({
            'name': filename,
            'datas': excel_data,
            'res_model': self._name,
            'res_id': self.id,
        })
        
        # Return download action
        return {
            'type': 'ir.actions.act_url',
            'url': f'/web/content/{attachment.id}?download=true',
            'target': 'self',
        }

    def download_spreadsheet(self):
        """Download the spreadsheet with the prepared data"""
        import io
        import xlsxwriter
        
        output = io.BytesIO()
        workbook = xlsxwriter.Workbook(output)
        worksheet = workbook.add_worksheet('Stock Card')

        # Formats
        header_format = workbook.add_format({
            'bold': True,
            'align': 'center',
            'bg_color': '#D3D3D3',
            'border': 1
        })

        # Write headers
        data = self.env.context.get('default_data', {})
        for col, header in enumerate(data.get('headers', [])):
            worksheet.write(0, col, header, header_format)
            worksheet.set_column(col, col, 15)

        # Write data
        for row_idx, row_data in enumerate(data.get('rows', []), start=1):
            for col_idx, cell_value in enumerate(row_data):
                worksheet.write(row_idx, col_idx, cell_value)

        workbook.close()
        output.seek(0)
        
        return {
            'type': 'ir.actions.act_url',
            'url': f'/web/content/{self._name}/{self.id}/spreadsheet?download=true',
            'target': 'self',
            'context': {
                'default_data': output.read(),
                'default_filename': self.env.context.get('filename', 'stock_card.xlsx')
            }
        }