# All Copyrights Reserved to Solid Super Solutions

from odoo import models, _
from odoo.exceptions import UserError
import io
from PyPDF2 import PdfFileReader, PdfFileWriter


class IrActionsReport(models.Model):
    _inherit = 'ir.actions.report'

    def _render_qweb_pdf(self, report_ref, res_ids=None, data=None):
        """
        Override to filter out journal entries without line_ids before PDF generation.
        This prevents empty PDF streams when printing multiple journal entries.
        """
        # Only handle our custom journal report
        if report_ref == 'sss_simple_journal_report.sss_simple_journal_report':
            # Filter out journal entries without line_ids BEFORE generating PDFs
            if res_ids:
                moves = self.env['account.move'].browse(res_ids)
                # Filter moves that have line_ids
                valid_moves = moves.filtered(lambda m: m.line_ids)
                
                # If some moves were filtered out, update res_ids
                if len(valid_moves) != len(moves):
                    res_ids = valid_moves.ids
                    
                    # If no valid moves, raise a user-friendly error
                    if not res_ids:
                        raise UserError(_(
                            'No valid journal entries to print. '
                            'All selected entries have no lines. Please select entries with lines.'
                        ))
        
        # Call parent method with filtered res_ids
        return super()._render_qweb_pdf(report_ref, res_ids=res_ids, data=data)

    def _render_qweb_pdf_prepare_streams(self, report_ref, data, res_ids=None):
        """
        Override to handle empty PDF streams when printing multiple journal entries.
        Filters out empty PDFs before combining them.
        """
        # Only handle our custom journal report
        if report_ref == 'sss_simple_journal_report.sss_simple_journal_report':
            # Filter out journal entries without line_ids BEFORE calling parent
            if res_ids:
                moves = self.env['account.move'].browse(res_ids)
                # Filter moves that have line_ids
                valid_moves = moves.filtered(lambda m: m.line_ids)
                
                # If some moves were filtered out, update res_ids
                if len(valid_moves) != len(moves):
                    res_ids = valid_moves.ids
                    
                    # If no valid moves, create a minimal valid PDF to avoid error
                    if not res_ids:
                        writer = PdfFileWriter()
                        output = io.BytesIO()
                        writer.write(output)
                        output.seek(0)
                        # Return in dictionary format
                        return {0: {'stream': output}}
            
            # Generate PDFs one by one to catch empty ones before base class reads them
            if res_ids and len(res_ids) > 1:
                collected_streams = {}
                for res_id in res_ids:
                    try:
                        # Generate PDF for single record
                        stream_result = super()._render_qweb_pdf_prepare_streams(
                            report_ref, data, res_ids=[res_id]
                        )
                        
                        # Base class returns dictionary: {res_id: {'stream': stream, ...}, ...}
                        if isinstance(stream_result, dict):
                            for key, value in stream_result.items():
                                if isinstance(value, dict) and 'stream' in value:
                                    stream = value['stream']
                                    
                                    # Check if stream is valid
                                    if isinstance(stream, io.BytesIO):
                                        current_pos = stream.tell()
                                        stream.seek(0, io.SEEK_END)
                                        size = stream.tell()
                                        stream.seek(current_pos)
                                        
                                        if size > 0:
                                            # Validate it's a valid PDF
                                            try:
                                                stream.seek(0)
                                                PdfFileReader(stream, strict=False)
                                                stream.seek(0)
                                                # Add to collected_streams
                                                collected_streams[key] = value
                                            except Exception:
                                                # Invalid PDF, skip it
                                                continue
                                        else:
                                            # Empty stream, skip it
                                            continue
                                else:
                                    collected_streams[key] = value
                        elif isinstance(stream_result, tuple):
                            # If it returns tuple, convert to dict format
                            stream, report_type = stream_result
                            if isinstance(stream, io.BytesIO):
                                current_pos = stream.tell()
                                stream.seek(0, io.SEEK_END)
                                size = stream.tell()
                                stream.seek(current_pos)
                                
                                if size > 0:
                                    try:
                                        stream.seek(0)
                                        PdfFileReader(stream, strict=False)
                                        stream.seek(0)
                                        collected_streams[res_id] = {'stream': stream}
                                    except Exception:
                                        continue
                    except Exception:
                        # Skip this record if it causes an error
                        continue
                
                if not collected_streams:
                    # All streams were empty, create a minimal valid PDF
                    writer = PdfFileWriter()
                    output = io.BytesIO()
                    writer.write(output)
                    output.seek(0)
                    # Return in dictionary format
                    return {res_ids[0]: {'stream': output}}
                
                return collected_streams
        
        # For single record or non-journal report, use parent method
        return super()._render_qweb_pdf_prepare_streams(report_ref, data, res_ids=res_ids)

